<?php
namespace MauticPlugin\CustomBrandingBundle\Command;

use Ramsey\Uuid\Uuid;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Doctrine\DBAL\Connection;

#[AsCommand(
    name: 'mautic:uuid:generate',
    description: 'Checks if the leads table has a uuid column'
)]
class GenerateUUIDsCommand extends Command
{
    private Connection $connection;

    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setDescription('The command generates UUIDs for entities that require them');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $tables = [
            'leads'
        ];

        foreach ($tables as $table) {
            $this->handleTable($table);
        }

        $output->writeln('UUIDs have been generated successfully.');
        return Command::SUCCESS;
    }

    /**
     * We need to generate UUIDs in batches to avoid memory issues on large datasets
     *
     * @param $table
     * @return void
     * @throws \Doctrine\DBAL\Exception
     */
    public function handleTable($table): void
    {
        $sql = "SHOW COLUMNS FROM $table LIKE 'uuid'";
        $result = $this->connection->fetchAssociative($sql);

        if (!$result) {
            return;
        }

        $batchSize = 1000;
        $offset    = 0;

        do {
            $sql = "SELECT id 
              FROM leads 
             WHERE uuid IS NULL OR uuid = '' 
             ORDER BY id ASC 
             LIMIT :limit OFFSET :offset";

            $rows = $this->connection->executeQuery($sql, [
                'limit'  => $batchSize,
                'offset' => $offset,
            ], [
                'limit'  => \PDO::PARAM_INT,
                'offset' => \PDO::PARAM_INT,
            ])->fetchAllAssociative();

            foreach ($rows as $row) {
                $uuid = Uuid::uuid4()->toString();

                $this->connection->update('leads', ['uuid' => $uuid], ['id' => $row['id']]);
            }

            $offset += $batchSize;

        } while (count($rows) > 0);
    }
}